// Copyright (C) 2014 to the present, Crestron Electronics, Inc.
// All rights reserved.
// No part of this software may be reproduced in any form, machine
// or natural, without the express written consent of Crestron Electronics.

/**
 *
 * \file        CresnetRconTimer.cpp
 *
 * \brief       Class that runs off timer and enqueues RCON packets into the console task
 *
 * \author      Adolfo Velasco
 *
 * \date        04/07/2014
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dm_nutypedefs.h"
#include "CresnetRconTimer.h"
#include "dm_os.h"
#include "dm_console.h"
#include "dm_errors.h"
#include "dm_cresnet.h"
#include "MemoryManager.h"
#include "CresnetConsole.hpp"

CresnetRconTimer* CresnetRconTimer::m_pTimer = NULL;

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
void CresnetRconTimer::Create( UINT8 bBufferCount ) 
{
    if ( !m_pTimer )
    {
        m_pTimer = new CresnetRconTimer(bBufferCount);
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
void CresnetRconTimer::Delete( void )
{
    if ( m_pTimer )
    {
        delete m_pTimer;
        m_pTimer = 0;
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
BOOL CresnetRconTimer::Add( UINT8* pPacket )
{
    if ( !m_pTimer || !pPacket )
        return FALSE;

    return ( m_pTimer->AddPacket(pPacket) );
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
void CresnetRconTimer::Timer( void )
{
    if ( !CresnetConsole || !MemMgr || !m_pTimer )
        return;

    if ( m_pTimer->m_hLock )
    {
        OsLock(m_pTimer->m_hLock);
    }

    UINT8* pPacket = m_pTimer->GetCurrentPacket();
    if ( pPacket )
    {
        //Attempt to forward the packet to the CresnetConsole class without any timeout
        if ( CresnetConsole->CresnetConsoleProcessRcon(pPacket, 0) )
        {
            //If packet was successfully enqueued, free the block and increment the read
            MemMgr->FreeBlock(pPacket);
            m_pTimer->IncrementRead();
        }
    }

    if ( m_pTimer->m_hLock )
    {
        OsUnlock(m_pTimer->m_hLock);
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
CresnetRconTimer::CresnetRconTimer( UINT8 bBufferCount ) :
m_hTimer(NULL),
m_bWriteIndex(0),
m_bReadIndex(0),
m_bBufferCount(0),
m_pPacketBuffer(NULL),
m_hLock(NULL)
{
    m_bWriteIndex = 0;
    m_bReadIndex = 0;

    if ( bBufferCount )
    {
        m_pPacketBuffer = (CresnetRconTimerEntry*) OsAllocMemory(sizeof(CresnetRconTimerEntry) * bBufferCount);
        if ( m_pPacketBuffer )
        {
            m_bBufferCount = bBufferCount;

            OsCreateLock(&m_hLock);

            m_hTimer = (UINT32) OsStartTimer(CRESNET_RCON_TIMER_PERIOD_MS, (void*)CresnetRconTimer::Timer, OS_TIMER_PERIODIC);
        }
    }

    //Check if creation failed
    if ( bBufferCount && (!m_pPacketBuffer || !m_hLock || !m_hTimer) )
    {
        //Log an error
        DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_CNET, ERR_CNET_SLAVE_RCON_TIMER_CREATE_FAILED, 0);
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
CresnetRconTimer::~CresnetRconTimer()
{
    if ( m_hTimer )
    {
        OsCancelTimer((void*)m_hTimer);
        m_hTimer = 0;
    }

    OsEnterCritical();

    if ( m_pPacketBuffer )
    {
        OsFreeMemory(m_pPacketBuffer);
        m_pPacketBuffer = 0;
        m_bBufferCount = 0;
    }

    if ( m_hLock )
    {
        OsDeleteLock(m_hLock);
        m_hLock = 0;
    }

    OsExitCritical();
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
BOOL CresnetRconTimer::AddPacket( UINT8* pPacket )
{
    BOOL bStatus = FALSE;

    if ( !pPacket || !m_bBufferCount || (m_bWriteIndex > m_bBufferCount-1) )
        return FALSE;

    if ( m_hLock )
    {
        OsLock(m_hLock);
    }

    //Check if we have space
    if ( m_bWriteIndex != GetPreviousIndex(m_bReadIndex) )
    {
        //Save the packet
        m_pPacketBuffer[m_bWriteIndex].pPacket = pPacket;

        //Increment to the next write location
        m_bWriteIndex = GetNextIndex(m_bWriteIndex);

        bStatus = TRUE;
    }

    if ( m_hLock )
    {
        OsUnlock(m_hLock);
    }

    return bStatus;
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
UINT8 CresnetRconTimer::GetNextIndex( UINT8 bCurrentIndex )
{
    if ( m_bBufferCount )
    {
        return ( bCurrentIndex >= m_bBufferCount-1 ? 0 : bCurrentIndex + 1 );        
    }
    else
    {
        return 0;
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
UINT8 CresnetRconTimer::GetPreviousIndex( UINT8 bCurrentIndex )
{
    if ( m_bBufferCount )
    {
        return ( bCurrentIndex > 0 ? bCurrentIndex-1 : m_bBufferCount-1 );        
    }
    else
    {
        return 0;
    }
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
BOOL CresnetRconTimer::IsTimerCreated( void )
{
    return( m_pTimer ? TRUE : FALSE );
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
UINT8* CresnetRconTimer::GetCurrentPacket( void )
{
    UINT8* pPacket = NULL;

    if ( m_bWriteIndex != m_bReadIndex )
    {
        pPacket = m_pPacketBuffer[m_bReadIndex].pPacket;
    }

    return pPacket;
}

/**
 * \author      Adolfo Velasco
 * \brief       
 * \date        04/07/2014
 * \param		void
 * \return    
 * \retval    
 */
void CresnetRconTimer::IncrementRead( void )
{
    m_bReadIndex = GetNextIndex(m_bReadIndex);
}


